﻿#include "proj.h"           // 包含 PROJ 库的头文件，提供地理坐标转换功能
#include <iostream>         // 包含标准输入输出流库
#include <iomanip>          // 包含用于格式化输出的库

using namespace std;        // 使用标准命名空间，简化 std::cout 等的调用

int main() {
    PJ_CONTEXT* C = nullptr; // 定义一个指针，指向 PROJ 上下文对象，用于管理多线程环境
    PJ* P = nullptr;         // 定义一个指针，指向投影变换对象
    PJ* norm = nullptr;      // 定义一个指针，指向标准化后的投影变换对象
    PJ_COORD a, b;           // 定义两个 PJ_COORD 结构体变量，用于存储坐标数据

    // 创建 PROJ 上下文
    C = proj_context_create(); // 创建一个新的 PROJ 上下文
    if (!C) {                 // 检查是否成功创建上下文
        cerr << "Failed to create PROJ context." << endl;
        return 1;             // 如果失败，输出错误信息并退出程序
    }

    // 创建从 EPSG:4326 到 UTM Zone 32N 的坐标转换对象
    P = proj_create_crs_to_crs(C,
                               "EPSG:4326",                           // 源 CRS (WGS84 地理坐标系)
                               "+proj=utm +zone=32 +datum=WGS84",    // 目标 CRS (UTM Zone 32N)
                               NULL);
    if (!P) { // 检查是否成功创建转换对象
        cerr << "Failed to create transformation object." << endl;
        proj_context_destroy(C); // 销毁上下文以释放资源
        return 1;               // 输出错误信息并退出程序
    }

    // 将转换对象标准化，确保输入坐标按照经度、纬度顺序
    norm = proj_normalize_for_visualization(C, P);
    if (!norm) { // 检查是否成功标准化转换对象
        cerr << "Failed to normalize transformation object." << endl;
        proj_destroy(P);        // 销毁原始转换对象
        proj_context_destroy(C); // 销毁上下文
        return 1;              // 输出错误信息并退出程序
    }

    proj_destroy(P); // 释放原始转换对象
    P = norm;        // 使用标准化后的转换对象

    // 定义一个地理坐标点：经度 12°E，纬度 55°N
    a = proj_coord(12, 55, 0, 0); // 初始化坐标，第三个和第四个参数通常为高程和时间，这里设置为 0

    // 执行正向转换（从地理坐标到 UTM 坐标）
    b = proj_trans(P, PJ_FWD, a); // 使用 PJ_FWD 标志进行正向转换
    cout << fixed << setprecision(6); // 设置浮点数输出为固定精度，保留六位小数
    cout << "East: " << b.xy.x << "; North: " << b.xy.y << endl; // 输出东向和北向坐标

    // 执行反向转换（从 UTM 坐标到地理坐标）
    b = proj_trans(P, PJ_INV, b); // 使用 PJ_INV 标志进行反向转换
    cout << "Longitude: " << b.lp.lam << "; Latitude: " << b.lp.phi << endl; // 输出经度和纬度

    // 清理资源
    proj_destroy(P); // 销毁转换对象
    proj_context_destroy(C); // 销毁上下文

    return 0; // 正常结束程序
}
